//------------------------------------------------------------------------------  
//title: Tundra Universe PCI-VME Kernel Driver
//version: Linux 0.8b
//date: September 1998                                                                
//designer: Michael Wyrick                                                      
//programmer: Michael Wyrick                                                    
//company: VMELinux Project http://www.vmelinux.org/
//platform: Linux 2.0.35 running on a Xycom XVME-655                                        
//language: GCC 2.7.2
//module: ca91c042
//------------------------------------------------------------------------------  
//  Purpose: Provide a Kernel Driver to Linux for the Universe PCI-VME Bridge
//           Universe model number ca91c042
//  Docs:                                                                       
//------------------------------------------------------------------------------  
#define MODULE
#include <stdio.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/major.h>
#include <linux/mm.h>
#include <linux/config.h>
#include <linux/fcntl.h>
#include <linux/unistd.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/proc_fs.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <asm/system.h>

#include "ca91c042.h"

#if LINUX_VERSION_CODE < 0x020100
#define	ioremap	vremap
#define	iounmap	vfree
#endif

//----------------------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------------------
static int uni_open(struct inode *inode,struct file *file);
static void uni_release(struct inode *inode,struct file *file);
static int uni_read(struct inode *inode,struct file *file,char *buf,int count);
static int uni_write(struct inode *inode,struct file *file,const char *buf,int count);
static int uni_ioctl(struct inode *inode,struct file *file,unsigned int cmd, unsigned long arg);
static int uni_procinfo(char *buf, char **start, off_t fpos, int lenght, int dummy);
static int uni_lseek(struct inode *inode,struct file *file,off_t offset,int whence);

//----------------------------------------------------------------------------
// Types
//----------------------------------------------------------------------------
static struct proc_dir_entry uni_procdir = {
  0,8,"ca91c042", S_IFREG | S_IRUGO,1,0,0,0,0, uni_procinfo  };

static struct file_operations uni_fops = 
{
  uni_lseek,
  uni_read,
  uni_write,
  NULL,         // uni_readdir,
  NULL,         // uni_select,
  uni_ioctl,
  NULL,         // uni_mmap,
  uni_open,
  uni_release 
};

static int aCTL[] = {LSI0_CTL, LSI1_CTL, LSI2_CTL, LSI3_CTL};
static int aBS[]  = {LSI0_BS,  LSI1_BS,  LSI2_BS,  LSI3_BS};
static int aBD[]  = {LSI0_BD,  LSI1_BD,  LSI2_BD,  LSI3_BD};
static int aTO[]  = {LSI0_TO,  LSI1_TO,  LSI2_TO,  LSI3_TO};

//----------------------------------------------------------------------------
// Vars and Defines
//----------------------------------------------------------------------------
#define UNI_MAJOR   	70
#define MAX_MINOR 	8
#define CONTROL_MINOR   8

#define MODE_UNDEFINED  0
#define MODE_PROGRAMMED 1
#define MODE_DMA	2

static int OkToWrite[MAX_MINOR + 1];        // Can I write to the Hardware

static int opened[MAX_MINOR + 1];
static int mode[MAX_MINOR + 1];		    // DMA or Programmed I/O

static unsigned long DMA[MAX_MINOR + 1];              // DMA Control Reg

// PCI bus stuff for playing
static int pci_found       = 0;
static unsigned char pcibus,pci_device_fn;
static unsigned short vendor, device;
static int status;
static char *baseaddr  = 0;
static char *image_ba[MAX_MINOR+1];	       // Base PCI Address

static unsigned int image_ptr[MAX_MINOR+1];
static unsigned int image_va[MAX_MINOR+1];
			      
// Status Vars
static unsigned int reads  = 0;
static unsigned int writes = 0;
static unsigned int ioctls = 0;

//----------------------------------------------------------------------------
//  uni_procinfo()
//----------------------------------------------------------------------------
static int uni_procinfo(char *buf, char **start, off_t fpos, int lenght, int dummy)
{
  char *p;
  unsigned int temp1,temp2,x;
  
  p = buf;
  p += sprintf(p,"Universe driver info:\n");

// Added back these statistics to Proc info - J. Huggins 2/25/98
  p += sprintf(p,"  Control Pointer = %04X\n",image_ptr[CONTROL_MINOR]);
  p += sprintf(p,"  Stats  reads = %i  writes = %i  ioctls = %i\n",
                 reads,writes,ioctls);

  temp1 = readl(baseaddr+PCI_BS);
//  p += sprintf(p,"  PCI_BS   = %08X\n",temp1);       
  temp1 = readl(baseaddr+PCI_CSR);
//  p += sprintf(p,"  PCI_CSR  = %08X\n",temp1);       
//  p += sprintf(p,"\n");

  for(x=0;x<4;x+=2) {
    temp1 = readl(baseaddr+aCTL[x]);
    temp2 = readl(baseaddr+aCTL[x+1]);
    p += sprintf(p,"  LSI%i_CTL = %08X    LSI%i_CTL = %08X\n",x,temp1,x+1,temp2);
    temp1 = readl(baseaddr+aBS[x]);
    temp2 = readl(baseaddr+aBS[x+1]);
    p += sprintf(p,"  LSI%i_BS  = %08X    LSI%i_BS  = %08X\n",x,temp1,x+1,temp2);
    temp1 = readl(baseaddr+aBD[x]);
    temp2 = readl(baseaddr+aBD[x+1]);
    p += sprintf(p,"  LSI%i_BD  = %08X    LSI%i_BD  = %08X\n",x,temp1,x+1,temp2);
    temp1 = readl(baseaddr+aTO[x]);
    temp2 = readl(baseaddr+aTO[x+1]);
    p += sprintf(p,"  LSI%i_TO  = %08X    LSI%i_TO  = %08X\n",x,temp1,x+1,temp2);
  }  
  
//  for(x=0;x<MAX_MINOR-1;x+=2)
  for(x=0;x<3;x+=2)
    p += sprintf(p,"  image_va%i   = %08X     image_va%i   = %08X\n",
      x,image_va[x],x+1,image_va[x+1]); 
  p += sprintf(p,"\nDriver Program Status:\n");  
      
  for(x=0;x<3;x+=2)
    p += sprintf(p,"  DMACTL %i    = %08lX DMACTL %i    = %08lX\n",
      x,DMA[x],x+1,DMA[x+1]); 	     
  for(x=0;x<3;x+=2)
    p += sprintf(p,"  OkToWrite %i = %1X        OkToWrite %i = %1X\n",
      x,OkToWrite[x],x+1,OkToWrite[x+1]); 
  for(x=0;x<3;x+=2)
    p += sprintf(p,"  Mode %i      = %1X        Mode %i      = %1X\n",
      x,mode[x],x+1,mode[x+1]); 

  p += sprintf(p,"\n");  
  
  return p - buf;
}

//----------------------------------------------------------------------------
//  register_proc()
//----------------------------------------------------------------------------
static void register_proc()
{
  proc_register_dynamic(&proc_root, &uni_procdir);
}

//----------------------------------------------------------------------------
//  unregister_proc()
//----------------------------------------------------------------------------
static void unregister_proc()
{
  proc_unregister(&proc_root, uni_procdir.low_ino);
}

//----------------------------------------------------------------------------
//  uni_open()
//----------------------------------------------------------------------------
static int uni_open(struct inode *inode,struct file *file)
{
  unsigned int minor = MINOR(inode->i_rdev);
  
  if (minor > MAX_MINOR) {
    return(-ENODEV);
  }
  
  if (!opened[minor]) {
    opened[minor] = 1;
    return(0);
  } else 
    return(-EBUSY);
}

//----------------------------------------------------------------------------
//  uni_release()
//----------------------------------------------------------------------------
static void uni_release(struct inode *inode,struct file *file)
{
  unsigned int minor = MINOR(inode->i_rdev);
  
  opened[minor] = 0;
}

//----------------------------------------------------------------------------
//  uni_lseek()
//----------------------------------------------------------------------------
static int uni_lseek(struct inode *inode,struct file *file,off_t offset,int whence)
{
  unsigned int minor = MINOR(inode->i_rdev);
  unsigned int toffset,base,paddr;
  
  if (whence == SEEK_SET) {	       
    if (minor == CONTROL_MINOR) {
      image_ptr[minor] = offset;
    } else {
      toffset = readl(baseaddr+aTO[minor]);    
      base    = readl(baseaddr+aBS[minor]);    
      paddr   = offset-toffset;
      image_ptr[minor] = (int)(image_ba[minor]+(paddr-base));
    }  
    return 0;
  } else if (whence == SEEK_CUR) {
    image_ptr[minor] += offset;
    return 0;
  } else
    return -1;  
}

//----------------------------------------------------------------------------
//  uni_read()
//----------------------------------------------------------------------------
static int uni_read(struct inode *inode,struct file *file,char *buf,int count)
{
  int x = 0;
  unsigned int v,numt,remain,tmp;
  char *temp = buf;
  unsigned int minor = MINOR(inode->i_rdev);  
  int p; 
  
  unsigned char  vc;
  unsigned short vs;
  unsigned int   vl;
  
  char *DMA_Buffer;
  unsigned int DMA_Buffer_Size = 0, 
               order = 0,
               a_size = 0;						
  int val,
      pci = 0,
      vme = 0,
      okcount = 0;
  
  if (minor == CONTROL_MINOR) {
    p = (int)image_ptr[minor];
    v = readl(baseaddr+p);    
    put_fs_long(v,temp);
  } else { 							      
    reads++;  	       	    
    if (OkToWrite[minor]) {
      if (mode[minor] == MODE_PROGRAMMED) {
        numt = count;
        remain = count;
      
        // Calc the number of longs we need
        numt = count / 4;
        remain = count % 4;
        for (x=0;x<numt;x++) {
          vl = readl(image_ptr[minor]);    
	  
          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount += 4;

          put_fs_long(vl,temp);
          image_ptr[minor]+=4;
          temp+=4;
        }  

        // Calc the number of Words we need
        numt = remain / 2;
        remain = remain % 2;
        for (x=0;x<numt;x++) {
          vs = readw(image_ptr[minor]);    

          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount += 2;

          put_fs_word(vs,temp);
          image_ptr[minor]+=2;
          temp+=2;
        }  
        
        for (x=0;x<remain;x++) {
          vc = readb(image_ptr[minor]);    

          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount++;

          put_fs_byte(vc,temp);
          image_ptr[minor]+=1;
          temp+=1;
        }  

      } else if (mode[minor] == MODE_DMA) {   
        // Wait for DMA to finish, This needs to be changed
        val = readl(baseaddr+DGCS);
 //       while (val & 0x00008000) 
 //         val = readl(baseaddr+DGCS);
	  
        // Setup DMA Buffer to read data into
	DMA_Buffer_Size = count + ((int)image_ptr % 8);         
	a_size = PAGE_SIZE;
	while (a_size < DMA_Buffer_Size) {
	  order++;
	  a_size <<= 1;
	}
	DMA_Buffer = (char *)__get_dma_pages(GFP_KERNEL,order);	 
	
        // VME Address						      
	vme = image_va[minor] + (image_ptr[minor] - (unsigned int)image_ba[minor]);  
	
	// PCI Address
	pci = virt_to_bus(DMA_Buffer);
	
	// Setup DMA regs
	writel(DMA[minor],baseaddr+DCTL);		// Setup Control Reg
	writel(count,baseaddr+DTBC);	     		// Count	    
	writel(pci,baseaddr+DLA); 			// PCI Address
	writel(vme,baseaddr+DVA);		 	// VME Address
	
//	printk("<<ca91c042 DMA READ C=%i P=%08X V=%08X >>\n",count,pci,vme);
	
	// Start DMA
	writel(0x80006F00,baseaddr+DGCS);		// GO
	
        // Wait for DMA to finish, This needs to be changed
        val = readl(baseaddr+DGCS);
        while (val & 0x00008000) 
          val = readl(baseaddr+DGCS);
	  
//	printk("<<ca91c042 DMA READ Finished DGCS = %08X>>\n",val);
	
        image_ptr[minor] += count;
	
	if (! (val & 0x00000800)) {  // An Error Happened
	  if (val & 0x00000400)
   	    printk("<<ca91c042 DMA PCI Bus Error>>\n");
	  if (val & 0x00000200)
   	    printk("<<ca91c042 DMA VME Bus Error>>\n");
	  if (val & 0x00000100)
   	    printk("<<ca91c042 DMA Protocol Error>>\n");
	  count = 0;	
	}
		
   	// Copy pages to User Memory
	memcpy_tofs(temp,DMA_Buffer,count);
	
	free_pages((unsigned long)DMA_Buffer,order);
      }           // end of MODE_DMA
    }             // end of OKtoWrite
  }  
  return(count);
}

//----------------------------------------------------------------------------
//  uni_write()
//----------------------------------------------------------------------------
static int uni_write(struct inode *inode,struct file *file,const char *buf,int count)
{
  int x,p;
  unsigned int numt,remain,tmp;
  char *temp = (char *)buf;
  unsigned int minor = MINOR(inode->i_rdev);
  
  unsigned char  vc;
  unsigned short vs;
  unsigned int   vl;
  
  char *DMA_Buffer;
  unsigned int DMA_Buffer_Size = 0, 
               order = 0,
               a_size = 0;						
  int val,
      pci = 0,
      vme = 0,
      okcount = 0;
  
  writes++;
  if (minor == CONTROL_MINOR) {
    vl = get_fs_long((unsigned long *)temp);
    p = (int)image_ptr[minor];
    writel(vl,baseaddr+p);
  } else { 
    if (OkToWrite[minor]) {
      if (mode[minor] == MODE_PROGRAMMED) {
        // Calc the number of longs we need
        numt = count;
        remain = count;
	
        numt = count / 4;
        remain = count % 4;
        for (x=0;x<numt;x++) {
          vl = get_fs_long((unsigned int *)temp);
          writel(vl,image_ptr[minor]);

          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount += 4;

          image_ptr[minor]+=4;
          temp+=4;
        }  

        // Calc the number of Words we need
        numt = remain / 2;
        remain = remain % 2;
    
        for (x=0;x<numt;x++) {
          vs = get_fs_word((unsigned short *)temp);
          writew(vs,image_ptr[minor]);
	  
          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount += 2;
	    
          image_ptr[minor]+=2;
          temp+=2;
        }  
      
        for (x=0;x<remain;x++) {
          vc = get_fs_byte((unsigned char *)temp);
          writeb(vc,image_ptr[minor]);
	  
          // Lets Check for a Bus Error
          tmp = readl(baseaddr+PCI_CSR);
          if (tmp & 0x08000000) {
            writel(tmp,baseaddr+PCI_CSR);
	    return(okcount);
          } else
	    okcount += 2;
	    
          image_ptr[minor]+=1;
          temp+=1;
        }  
	
        // Lets Check for a Bus Error
        tmp = readl(baseaddr+PCI_CSR);
        if (tmp & 0x08000000) {  // S_TA is Set
          writel(0x08000000,baseaddr+PCI_CSR);
          count = 0;
        }

      }	else if (mode[minor] == MODE_DMA) {
        // Wait for DMA to finish, This needs to be changed
        val = readl(baseaddr+DGCS);
 //       while (val & 0x00008000) 
 //         val = readl(baseaddr+DGCS);
	  
        // Setup DMA Buffer to write data into
	DMA_Buffer_Size = count + ((int)image_ptr % 8);         
	a_size = PAGE_SIZE;
	while (a_size < DMA_Buffer_Size) {
	  order++;
	  a_size <<= 1;
	}
	DMA_Buffer = (char *)__get_dma_pages(GFP_KERNEL,order);	 
	
   	// Copy User Memory into buffer
	memcpy_fromfs(DMA_Buffer,temp,count);
	
        // VME Address						      
	vme = image_va[minor] + (image_ptr[minor] - (unsigned int)image_ba[minor]);  
	
	// PCI Address
	pci = virt_to_bus(DMA_Buffer);
	
	// Setup DMA regs
	writel(DMA[minor]|0x80000000,baseaddr+DCTL);	// Setup Control Reg
	writel(count,baseaddr+DTBC);	     		// Count	    
	writel(pci,baseaddr+DLA); 			// PCI Address
	writel(vme,baseaddr+DVA);		 	// VME Address
	
//	printk("<<ca91c042 DMA WRITE C=%i P=%08X V=%08X >>\n",count,pci,vme);
	
	// Start DMA
	writel(0x80006F00,baseaddr+DGCS);		// GO
	
        // Wait for DMA to finish, This needs to be changed
        val = readl(baseaddr+DGCS);
        while (val & 0x00008000) 
          val = readl(baseaddr+DGCS);
	  
//	printk("<<ca91c042 DMA WRITE Finished DGCS = %08X>>\n",val);
	
        image_ptr[minor] += count;
	
	if (! (val & 0x00000800)) {  // An Error Happened
//	  if (val & 0x00000400)
//   	    printk("<<ca91c042 DMA PCI Bus Error>>\n");
//	  if (val & 0x00000200)
//   	    printk("<<ca91c042 DMA VME Bus Error>>\n");
//	  if (val & 0x00000100)
//   	    printk("<<ca91c042 DMA Protocol Error>>\n");
	  count = 0;	
	}
	
	free_pages((unsigned long)DMA_Buffer,order);
      }           // end of MODE_DMA
    }  //OKtoWrite
  }  
  return(count);
}

//----------------------------------------------------------------------------
//  uni_ioctl()
//----------------------------------------------------------------------------
static int uni_ioctl(struct inode *inode,struct file *file,unsigned int cmd, unsigned long arg)
{
  unsigned int minor = MINOR(inode->i_rdev);
  unsigned int sizetomap = 0, to = 0, bs = 0;
  
  ioctls++;
  switch (cmd) {
    case IOCTL_SET_CTL:
      writel(arg,baseaddr+aCTL[minor]);
      // Lets compute and save the DMA CTL Register
      DMA[minor] = arg & 0x00FFFF00;
      break;
      	    
    case IOCTL_SET_MODE:
      mode[minor] = arg;
      break;
      
    // Lets Map the VME Bus to the PCI Bus
    //
    // << NOTE >> BD Must already be set before you call this!!!!
    //
    //
    case IOCTL_SET_BS:
      writel(arg,baseaddr+aBS[minor]);
      if (image_ba[minor])
        iounmap(image_ba[minor]);
  
      // This uses the BD Register to Find the size of the Image Mapping		
      sizetomap = readl(baseaddr+aBD[minor]);
      sizetomap -= arg;
      		
      image_ba[minor] = (char *)ioremap(arg,sizetomap);
      if (!image_ba[minor]) {
        OkToWrite[minor] = 0;     
        return -1;
      }
      image_ptr[minor] = (int)image_ba[minor];
      OkToWrite[minor] = 1;
      
      // Calculate the VME Address
      bs = readl(baseaddr+aBS[minor]);
      to = readl(baseaddr+aTO[minor]);
      image_va[minor]  = bs + to;
      break;
      
    case IOCTL_SET_BD:
      writel(arg,baseaddr+aBD[minor]);
      break;  
    case IOCTL_SET_TO:
      writel(arg,baseaddr+aTO[minor]);
      				      
      // Calculate the VME Address
      bs = readl(baseaddr+aBS[minor]);
      to = readl(baseaddr+aTO[minor]);
      image_va[minor]  = bs + to;
      break;  
    default:
      if (cmd < 0x1000) {  // This is a Register value so write to it.
        writel(arg,baseaddr+cmd);
      }  
      break;  
  }
  return(0);
}

//----------------------------------------------------------------------------
//  init_module()
//----------------------------------------------------------------------------
int init_module(void)
{
  int x;
  unsigned int temp;
  unsigned int ba;

  printk("Tundra Universe PCI-VME Bridge Driver 0.8b 980925\n");
  printk("  Copyright 1997-1998, Michael J. Wyrick and the VMELinux Project\n");
    
  if (pcibios_present()) {
    if (!pcibios_find_device(0x10E3,0x0000,0,&pcibus,&pci_device_fn)) {
      pci_found = 1;
      printk("  Universe device found at:  pcibus = %02X  devicefn = %02X\n",
        pcibus,pci_device_fn);
      
      // Lets turn Latency off
      pcibios_write_config_dword(pcibus,pci_device_fn,PCI_MISC0,0);
      
      // Display PCI Registers  
      pcibios_read_config_word(pcibus,pci_device_fn,PCI_VENDOR_ID,&vendor);
      pcibios_read_config_word(pcibus,pci_device_fn,PCI_DEVICE_ID,&device);
      pcibios_read_config_dword(pcibus,pci_device_fn,PCI_CSR,&status);
      printk("  Vendor = %04X  Device = %04X  Status = %08X\n",
        vendor,device,status);
      pcibios_read_config_dword(pcibus,pci_device_fn,PCI_MISC0,&temp);
      printk("  Misc0 = %08X\n",temp);
      pcibios_read_config_dword(pcibus,pci_device_fn,PCI_CLASS,&temp);
      printk("  Class = %08X\n",temp);
      
      // Setup Universe Config Space
      // This is a 4k wide memory area that need to be mapped into the kernel
      // virtual memory space so we can access it.
      pcibios_write_config_dword(pcibus,pci_device_fn,PCI_BS,CONFIG_REG_SPACE);
      pcibios_read_config_dword(pcibus,pci_device_fn,PCI_BS,&ba);        
      baseaddr = (char *)ioremap(ba,4096);
      if (!baseaddr) {
        printk("  vremap failed to map Universe to Kernel Space.\r");
        return 1;
      }
   
      // Check to see if the Mapping Worked out
      temp = readl(baseaddr);
      printk("  Read via mapping, PCI_ID = %08X\n",temp);       
      if (temp != 0x000010E3) {
        printk("  Universe Chip Failed to Return PCI_ID in Memory Map.\n");
        return 1;
      }
      
      // OK, Every this is ok so lets turn off the windows
      writel(0x00800000,baseaddr+LSI0_CTL);     
      writel(0x00800000,baseaddr+LSI1_CTL);     
      writel(0x00800000,baseaddr+LSI2_CTL);     
      writel(0x00800000,baseaddr+LSI3_CTL);     
      
    } else {
      printk("  Universe device not found on PCI Bus.\n");
      return 1;
    }
  } else {
    printk("  HEY - You can not run a PCI device on a non-pci system.");
    return 1;
  }
     
  if (register_chrdev(UNI_MAJOR, "uni", &uni_fops)) {
    printk("  Error getting Major Number for Drivers\n");
    iounmap(baseaddr);
    return(1);
  } else {
    printk("  Tundra Loaded.\n"); 
  }
  
  register_proc();
 
  for(x=0;x<MAX_MINOR+1;x++) {
    image_ba[x]  = 0;	     		// Not defined
    image_ptr[x] = 0;	     		// Not defined
    opened[x]    = 0;	     		// Closed
    OkToWrite[x] = 0;	     		// Not OK
    mode[x]      = MODE_UNDEFINED;   	// Undefined
  }  
 
  return 0;
}

//----------------------------------------------------------------------------
//  cleanup_module()
//----------------------------------------------------------------------------
void cleanup_module(void)
{	       
  int x;
  
  for (x=1;x<MAX_MINOR+1;x++) {
    if (image_ba[x])
      iounmap(image_ba[x]);
    }    
  iounmap(baseaddr);
  unregister_proc();
  unregister_chrdev(UNI_MAJOR, "uni");
}

